! ====================
  program covfunc_lsq
! ====================

! simple program to demonstrate LSQ estimation of covariance function
! form a matrix of covariance components, using Legendre polynomials
! as basis functions
!                                                           @C K.Meyer

  implicit none
  integer                              :: kq, kqq, nq, nqq, iq, ii, ipre, i, j
  real(8), dimension(:), allocatable   :: ages
  real(8), dimension(:,:), allocatable :: clgnd, phi, sigma, kmat
  logical                              :: legendr, lex
  character(len=30)                    :: fname

! specify options
  write(*,*)'no. of grid points ? '; call option (nq,1,999)
  nqq = nq*(nq+1)/2;  allocate( ages(nq), stat=ii); if( ii/= 0) stop 'all'
  write(*,*)'give values of control variable (ascending order)'
  write(*,*)' ';  write(*,*)'dimension of grid given =',nq
  ipre = -10000
  do iq = 1, nq
     write(*,*)' value',iq; call option( ii, ipre+1,99999); ages(iq) = ii
     ipre = ii
  end do
  do iq = 1, nq
     write(*,*)'value of control variable',iq,ages(iq)
  end do
  write(*,*)' '
  write(*,*)'no. of random regression coefficients ?'; call option(kq,1,nq)
  kqq = kq*(kq+1)/2
  allocate( sigma(nq,nq), kmat(kq,kq), stat = ii); if( ii/= 0) stop 'alloc sig'
  sigma = 0.d0; kmat = 0.d0
  write(*,*)'give name of file with (co)variances for grid points'
  write(*,*)' (this must contain the *upper* triangle row-wise) '
  read(*,'(a)') fname;  inquire( file = trim(fname), exist=lex )
  if( .not.lex ) stop 'file does not exist'
  open(1, file = trim(fname),status='old',form='formatted',action='read')
  read(1,*,iostat=ii) (( sigma(i,j),j=i,nq ),i=1,nq )
  if( ii/= 0) then
      write(*,*)'error reading covariance matrix '; stop
  end if
  write(*,*)' '; write(*,*)'matrix of covariance components'
  do i = 1, nq
     do j = i, nq
        sigma(j,i) = sigma(i,j)
     end do
     write(*,'(i3,(t4,8g14.7))') i, sigma(i,:)
  end do
  legendr = .true.

! obtain least-squares estimates of covaraince function
  if( legendr ) then
      allocate( clgnd(kq,kq), phi(nq, kq), stat = ii)
      if( ii/= 0 ) stop 'alloc clgnd'
      call legendre( kq, nq, ages, clgnd, phi )
  end if
  call kmat_lsq( kq, nq, phi, sigma, kmat)

  write(*,*)'Estimates of covariance matrix among RR coefficients'
  do i=1,kq
     write(*,'(i2,(t4,8g13.7))') i, kmat(i,:)
  end do

  end program covfunc_lsq

! ================================================
  subroutine kmat_lsq (kfit, na, phi, sigma, kmat)
! =================================================

  implicit none
  integer, intent(in)                        :: kfit, na
  real(8), dimension(na,kfit), intent(in)    :: phi
  real(8), dimension(na,na), intent(in)      :: sigma
  real(8), dimension(kfit,kfit), intent(out) :: kmat

  real(8), dimension(:,:), allocatable :: xmat
  real(8), dimension(:), allocatable   :: work, w1, w2, xkk
  integer, dimension(:), allocatable   :: iwork
  real(8)                              :: detx, xx
  integer                              :: kk, ii, i, j, ij, m, n, mn, k, irank
  integer, external                    :: ihmssf

  kk=kfit*(kfit+1)/2
  allocate (xmat(na*(na+1)/2,kk),work(kk*(kk+1)/2),w1(kk), w2(na*(na+1)/2), &
&           xkk(kk), stat=ii);  if(ii>0)stop 'alloc kmat_lsq'
      
! design matrix x
  xmat=0.d0
  ij=0
  do i=1,na
     do j=i,na
        ij=ij+1
        do m=1,kfit
           do n=1,kfit
              mn=ihmssf(m,n,kfit)
              xmat(ij,mn)=xmat(ij,mn) + phi(i,m)*phi(j,n)
           end do
        end do
     end do
  end do
! ... coefficient matrix x'x
  ij=0
  do i = 1, kk
     do j = i, kk
        ij=ij+1; work(ij)= dot_product( xmat(:,i), xmat(:,j) )
     end do
  end do
! ... rhs x'vec(v)
  call full2half( sigma, w2, na )
  do i = 1, kk
     w1(i) = dot_product( xmat(:,i),w2 )
  end do
! ... inverse
  call dkmxhf( work, detx, 1.d-12, irank, kk, .false.)
! ... lsq-estimate
  ij=0
  do ij=1,kk
     xx=0.d0
     do m=1,kk
        xx=xx+work(ihmssf(ij,m,kk))*w1(m)
     end do
     xkk(ij)=xx
  end do
  call half2full( xkk, kmat, kfit )
  deallocate (xmat,work,w1, w2, xkk, stat=ii); if(ii/=0)stop 'deall kmatlsq'

  return
  end subroutine kmat_lsq

! ===============================================
  subroutine legendre (nc, na, ages, clgnd, phi )
! ===============================================

  implicit none
  integer, intent(in)                    :: nc, na
  real(8), dimension(na), intent(in)     :: ages
  real(8), dimension(nc,nc), intent(out) :: clgnd
  real(8), dimension(na,nc), intent(out) :: phi
  real(8), dimension(na)                 :: astar
  real(8)                                :: c1, c2, cc, binfac, pp, aa, aamin,&
&                                           xlow=-1.d0, xupp=1.d0
  integer                                :: iord, jj, i, j, kk, m, iq, ifit
  
! evaluate coefficients
  clgnd = 0.d0
  do iord = 0, nc-1
     if( iord == 0 ) then
         c1 = 1.d0; c2 = 0.5d0
      else
         c1 = 2.d0**iord; c2 = iord+0.5d0
      end if
      cc=dsqrt(c2)/c1; jj = iord/2
      do m = jj, 0, -1
         i = 1+iord-2*m; j = 2*(iord-m); kk=1; if(m > 0) kk = (-1)**m
         clgnd(i,iord+1) = kk*cc*binfac(iord,m)*binfac(j,iord)
      end do
  end do

! standardise ages
  astar = ages; aa= (xupp-xlow)/(astar(nc)-astar(1))
  aamin = astar(1); astar = xlow + aa*( astar -aamin )

! set up phi matrix
  do ifit = 1, nc
     do iq = 1, na
        aa = astar(iq); pp = clgnd(1,ifit)
        do j = 2, ifit
           pp = pp + clgnd(j,ifit)*aa**(j-1)
        end do
        phi(iq,ifit) = pp
     end do
  end do
  do i = 1,na
     print *,'phi',i,phi(i,:)
  end do
  return
  end subroutine legendre

! =============================
  real(8) function binfac (n,m)
! =============================

  implicit none
  integer, intent(in) :: n,m
  real(8)             :: factrl,b1,b2,b3
  integer             :: n1

  if(m.eq.0.or.m.eq.n)then
     binfac=1.d0
  else
     b1=factrl(n); n1=n-m; b2=factrl(n1); b3=factrl(m)
     binfac=b1/(b2*b3)
  end if

  return
  end function binfac

! ===========================
  real(8) function factrl (n)
! ===========================

  implicit none
  integer, intent(in) :: n
  real(8)             :: ff
  integer             :: i

  ff = 1.d0
  do i = 2,n
     ff = ff*i
  end do
  factrl=ff

  return
  end function factrl

! =====================================
  subroutine option( iopt, iomin, iomax)
! ======================================

!     purpose : routine to read in option (numerical value) and
!               check that it is in the correct range
!------------------------------------------------------------km--6/89--

 implicit none
 integer, intent(in)  :: iomin, iomax
 integer, intent(out) :: iopt
 character(len=20)    :: a
 real(8)              :: xopt

 do
    read(*,'(a)') a; call chkdigit( a, iopt, xopt, 20, 1 )
    if( (iopt < iomin) .or. (iopt > iomax) )then
         write(*,'(1x,a,i5,3x,a,i6,a)') &
&                 'invalid option given - permissible range is',       &
&                  iomin,'to',iomax,' !!'
         write(*,*)'value specified was',iopt,'  try again: option ? '
    else
         return
    end if
  end do

  end subroutine option

! ===========================================
  subroutine chkdigit( a, iopt, xopt, n, jj )
! ===========================================

  implicit none
  integer, intent(in)             :: jj, n
  character(len=n), intent(inout) :: a
  integer, intent(out)            :: iopt
  real(8), intent(out)            :: xopt

  integer                         :: mm, nvalid, i, j, nn
  integer, dimension(2)           :: ll = (/ 12, 18 /)
  logical                         :: invalid
  character(len=15)               :: fmt
  character(len=1), dimension(18) :: digit = (/' ','0','1','2','3','4','5', &
&                                              '6','7','8','9','-','.','+', &
&                                              'e','e','d','d'/)

! option is blank ...
  do while( len_trim(a) < 1 )
     write(*,*)'option must be non-blank - again ...'
     read(*,'(a)') a
  end do
  nn = len_trim(a)

  invalid = .true.
1 do while( invalid )
     nvalid = 0; mm = 0
     do i = 1, nn
        do j = 1, ll(jj)
           if( a(i:i) == digit(j) )then
               if( j.ge.12 )then
                   if( (mm.eq.j) .or. (j.gt.14.and.mm.gt.14) ) exit
                   mm=j
               end if
               go to 9
           end if
        end do
        write(*,*)'non-digit encountered : ',a
        write(*,*)'try again - option ?'; read(*,'(a)')a
        go to 1
9       nvalid = nvalid + 1
     end do
     if( nvalid == nn ) invalid = .false.
  end do
  if( jj == 1 ) then
      write(fmt,'(a,i2,a)') '(bn,i',nn, ')'; read(a,fmt) iopt      
  else
      write(fmt,'(a,i2,a)') 'bn,g',nn, '.0)'; read(a,fmt) xopt      
  end if

  return
  end subroutine chkdigit

! ====================================
  subroutine full2half( hh, ww, mfit)
! ====================================

  implicit none
  integer, intent(in)                              :: mfit
  real(8), dimension(mfit, mfit), intent(in)       :: hh
  real(8), dimension(mfit*(mfit+1)/2), intent(out) :: ww
  integer                                          :: i, j, ij

  ij = 0
  do i = 1, mfit
     do j = i, mfit
        ij = ij + 1; ww(ij) = hh(j,i)
     end do
  end do

  return
  end subroutine full2half

! ====================================
  subroutine half2full( ww, hh, mfit)
! ====================================

  implicit none
  integer, intent(in)                              :: mfit
  real(8), dimension(mfit, mfit), intent(out)      :: hh
  real(8), dimension(mfit*(mfit+1)/2), intent(in)  :: ww
  integer                                          :: i, j, ij

  ij = 0
  do i = 1, mfit
     do j = i, mfit
        ij = ij + 1; hh(j,i) = ww(ij); hh(i,j) = ww(ij)
     end do
  end do

  return
  end subroutine half2full

! =======================================================================
  subroutine dkmxhf( a, det, zero, nrank, n, iopt)
! =======================================================================

  implicit none
  integer, intent(in)                          :: n
  logical, intent(in)                          :: iopt
  real(8), dimension(n*(n+1)/2), intent(inout) :: a
  real(8), intent(in)                          :: zero
  real(8), intent(out)                         :: det
  integer, intent(out)                         :: nrank
  real(8), dimension(n)                        :: v, w
  integer, dimension(n)                        :: iflag
  real(8)                                      :: ww, xx, dmax, &
&                                                 amax, bmax, dimax
  integer                                      :: ii,i1,n1,il,imax, &
&                                                 imaxm1,imaxp1,ij,j,i,neg

!     matrix is a scalar
      if(n.eq.1)then
         if(a(1).gt.zero)then
            det=dlog( a(1) ); a(1)=1.d0/a(1); nrank=1
         else
            a(1)=0.d0; nrank=0; det=0.d0
         end if
         return
      end if

!     initialize
      neg=0; det=0.d0; iflag=0;  n1=n+1

!     pick out diagonal elements
      ii=-n
      do i=1,n
         ii=ii+n1; w(i)=a(ii); ii=ii-i
      end do


!     gaussian elimination steps
      do 100 ii=1,n

!     find diag. element with largest absolute value (pivot)
      dmax=0.d0;  amax=0.d0
      do i=1,n
         if(iflag(i).ne.0) cycle
         bmax=dabs(w(i))
         if(bmax.gt.amax)then
            dmax=w(i); amax=bmax; imax=i
         end if
      end do

!     check for singularity
      if(amax.le.zero)go to 11
      if(iopt .and. amax.lt.1.d-5) print *,'small pivot ',ii,dmax
!     set flag
      iflag(imax)=ii
!     accumulate log determinant
      det=det+dlog(amax)
      if(dmax.lt.0.d0)then
         neg=neg+1
         if(iopt)print *,'negative pivot, ignore sign for log det' &
     &                       ,ii,imax,dmax
      end if

      dimax = 1.d0/dmax; imaxm1 = imax-1; imaxp1 = imax+1

!     pick out elements for row imax
      il=imax-n
      do i=1,imaxm1
         il=il+n1-i; v(i)=a(il)
      end do
      il=il+n1-imax
      do i=imaxp1,n
         il=il+1; v(i)=a(il)
      end do

!     transform matrix
      ij=0
      do i=1,imaxm1
      ww = v(i)
      if(ww /= 0.d0)then
         xx = ww*dimax
         ij = ij+1; w(i) = w(i) - ww*xx
         do j=i+1,imaxm1
            ij=ij+1
            if( v(j) /= 0) a(ij) = a(ij)-xx*v(j)
         end do
!        element a(i,imax)
         ij=ij+1; a(ij) = xx
         do j = imaxp1,n
            ij=ij+1
            if( v(j) /= 0) a(ij) = a(ij)-xx*v(j)
         end do
      else
         ij=ij+n1-i
      end if
      end do

!     row imax
      ij=ij+1; w(imax)=-dimax
      do j = imaxp1,n
         ij=ij+1; a(ij)=v(j)*dimax
      end do

      do i=imaxp1,n
         ww=v(i)
         if(ww /= 0.d0)then
            xx=ww*dimax; ij=ij+1; w(i)=w(i)-ww*xx
            do j=i+1,n
               ij=ij+1; a(ij)=a(ij)-xx*v(j)
            end do
         else
            ij=ij+n1-i
         end if
      end do
 100  continue

!     store diagonals back & reverse signs
      ij=0
      do i=1,n
         ij=ij+1; ww=-w(i); w(i)=ww; a(ij)=ww
         do j=i+1,n
            ij=ij+1; a(ij)=-a(ij)
         end do
      end do
      nrank=n

 300  if(iopt .and. neg.gt.0)print *,'no. of negative pivots =', neg
      if( iopt .and. mod(neg,2) /= 0 ) write(*,*)'dkmxhf : uneven no. of negative pivots',&
&                          ' - log determinant not correct !'
      return

!      matrix not of full rank, return generalised inverse
   11 nrank=ii-1; ij=0

      do i=1,n
      if(iflag(i) == 0)then
!        ... zero out row/column
         w(i)=0.d0
         do j=i,n
            ij=ij+1;  a(ij)=0.d0
         end do
      else
         ij=ij+1
         ww=-w(i); w(i)=ww; a(ij)=ww
         do j=i+1,n
            ij=ij+1
            if(iflag(j).ne.0)then
               a(ij)=-a(ij)
            else
               a(ij)=0.d0
            end if
         end do
      end if
      end do

      if(iopt)print 15,n,nrank
   15 format(' generalised inverse of matrix with order =',i5,'   and rank =',i5)
      go to 300
      end subroutine dkmxhf

! ===============================
  integer function ihmssf(i,j,n)
! ===============================

  implicit none
  integer, intent(in) :: i, j, n
  integer             :: i1, j1

  if(i.le.j)then
     i1=i-1; ihmssf=n*i1-i*i1/2+j
  else
     j1=j-1; ihmssf=n*j1-j*j1/2+i
  end if

  return
  end function ihmssf
